####################################################################
# Asymptotic variance of Reyment and Voinov and Nikulin estimates 
# (S and raw MCD estimators)
# /!\ the assumed underlying distribution  (used to compute the constants and expectations) is normal
####################################################################

# Files needed to run this script: 
# - MCV_consist
# - consistMCD
# - consistS
# - fastS_consistency
# - fastS_normality


# --------------- Main functions 
ASV_Reyment<-function(mu,sigma, estim="class",bdp=0.25){
  # INPUTS:
  # mu    : location estimate
  # sigma : scatter estimate
  # estim : location and scatter estimators 
  #     'class' : sample estimate
  #     'MCD'   : MCD estimate (raw)
  #     'S'     : S estimate with Tukey's biweight function
  # bdp : breakdown point of the above estimator
  
  # OUTPUT
  # ASV
  
  p<-length(mu)
  g<- cvreyment(mu,sigma)
  alpha<- bdp
  
  if(estim=="class"){
    ASV <- g^2/(2*p)*( 1+ (2*p)/(mu%*%mu)^2 * (mu%*%sigma%*%mu) )
    
  }else if(estim=="MCD"){
    qalpha<- qchisq(1-alpha,p)
    calpha<-(1-alpha)/pchisq(qalpha,p+2)
    c2<- -pchisq(qalpha,p+2)/2
    c3<- -pchisq(qalpha,p+4)/2
    b1<- - 2*calpha*c3/(1-alpha)
    b2<- 0.5 + calpha/(1-alpha)* (c3- qalpha/p*(c2+(1-alpha)/2) )
    C1<- 1/pchisq(qalpha,p+2)
    C2<- -1/ (2*c3)
    C3<- - b2/b1 *calpha/(1-alpha) 
    C4<- calpha/(1-alpha)*qalpha/p
    C5<- -(calpha*qalpha/p -1)
    
    ASV<- g^2/4 *(1/p* (C2-p*C3)^2*(p+2)*pchisq(qalpha,p+4) -2 *(C2-p*C3)*(C4+C5) *pchisq(qalpha,p+2)+ (C4^2+2*C4*C5)*(1-alpha) + C5^2+ 4/(mu%*%mu)^2 *C1^2 *(mu%*%sigma%*%mu) *pchisq(qalpha,p+2))
  }else if (extim=="S"){
    c <- csolve.bw.asymp(p,bdp)
    b<- erho.bw(p,c)
    bet<- chi.int(p,0,c) + 1/p* (- 2*(p+2)/c^2*chi.int(p,2,c) + (p+4)/c^4*chi.int(p,4,c) )
    g1<- 1/(p+2) *( (p+2)*chi.int(p,2,c) - 2/c^2*(p+4)*chi.int(p,4,c) + 1/c^4*(p+6)*chi.int(p,6,c))
    g3<- chi.int(p,2,c) - 2/c^2*chi.int(p,4,c) + 1/c^4*chi.int(p,6,c)
    C1<- p/g1
    C2<- -2/g3
    C3<- C1/p
    C4<- -b*C2 
    C5<- 1/bet
    B<- chi.int(p,4,c) - 4/c^2*chi.int(p,6,c) + 6/c^4*chi.int(p,8,c) - 4/c^6*chi.int(p,10,c) + 1/c^8*chi.int(p,12,c)
    D<- chi.int(p,2,c) - 4/c^2*chi.int(p,4,c) + 6/c^4*chi.int(p,6,c) - 4/c^6*chi.int(p,8,c) + 1/c^8 *chi.int(p,10,c)
    E<- chi.int(p,4,c)/4 - 1/(2*c^2)*chi.int(p,6,c) + 5/(12*c^4)*chi.int(p,8,c) - 1/(6*c^6)*chi.int(p,10,c) +1/(36*c^8)*chi.int(p,12,c) + c^4/36* chi.int2(p,0,c)
    
   ASV<- g^2/4*(C2^2*E -C4^2 + (4/p)*(mu%*%sigma%*%mu)/(mu%*%mu)^2 *C5^2*D)
  }else stop("This is not  valid estim value. The possible values are: 'class', 'MCD' or 'S'")
  return(ASV)
}

ASV_VN<-function(mu,sigma, estim="class",bdp=0.25){
  # INPUTS:
  # mu    : location estimate
  # sigma : scatter estimate
  # estim : location and scatter estimators 
  #     'class' : sample estimate
  #     'MCD'   : MCD estimate (raw)
  #     'S'     : S estimate with Tukey's biweight function
  # bdp : breakdown point of the above estimator
  
  # OUTPUT
  # ASV
 
  p<-length(mu)
  g<- cvvoinov(mu,sigma)
  alpha<- bdp
  
  if(estim=="class"){
    ASV<- g^2/2 +g^4
    
  }else if(estim=="MCD"){
    qalpha<- qchisq(1-alpha,p)
    calpha<-(1-alpha)/pchisq(qalpha,p+2)
    c2<- -pchisq(qalpha,p+2)/2
    c3<- -pchisq(qalpha,p+4)/2
    b1<- -2*calpha*c3/(1-alpha)
    b2<- 1/2 + calpha/(1-alpha)* (c3- qalpha/p*(c2+(1-alpha)/2) )
    C1<- 1/pchisq(qalpha,p+2)
    C2<- -1/ (2*c3)
    C3<- - b2/b1 *calpha/(1-alpha) 
    C4<-  calpha/(1-alpha)*qalpha/p
    C5<- -(calpha*qalpha/p -1)
    
    integrand<-function(x,mu,sigma,alpha){
      p<-length(mu)
      qalpha<- qchisq(1-alpha,p)
      distx<-Mahaladist(x,mu,sigma)
      central<-t(x)-mu
      return( as.numeric((distx^2< qalpha)*(colSums(solve(sigma,central)*mu))^4))
    }	
    ech<- rmvnorm(100000,mu,sigma)
    I<-mean(integrand(ech,mu=mu,sigma=sigma,alpha=alpha))
    
    ASV<- g^2/4*(g^4*C2^2*I + C3*(p*C3-2*C2)*(p+2)*pchisq(qalpha,df=p+4) + 
                   ( 2*(C4+C5)*(p*C3-C2) + 4* g^2*C1^2)*pchisq(qalpha,df=p+2) + C4*(C4+2*C5)*(1-alpha) + C5^2)
    
  }else if(estim=="S"){
    c <- csolve.bw.asymp(p,bdp)
    b<- erho.bw(p,c)
    bet<- chi.int(p,0,c) + 1/p* (- 2*(p+2)/c^2*chi.int(p,2,c) + (p+4)/c^4*chi.int(p,4,c) )
    g1<- 1/(p+2) *( (p+2)*chi.int(p,2,c) - 2/c^2*(p+4)*chi.int(p,4,c) + 1/c^4*(p+6)*chi.int(p,6,c))
    g3<- chi.int(p,2,c) - 2/c^2*chi.int(p,4,c) + 1/c^4*chi.int(p,6,c)
    C1<- p/g1
    C2<- -2/g3
    C3<- C1/p
    C4<- -b*C2 
    C5<- 1/bet
    B<- chi.int(p,4,c) - 4/c^2*chi.int(p,6,c) + 6/c^4*chi.int(p,8,c) - 4/c^6*chi.int(p,10,c) + 1/c^8*chi.int(p,12,c)
    D<- chi.int(p,2,c) - 4/c^2*chi.int(p,4,c) + 6/c^4*chi.int(p,6,c) - 4/c^6*chi.int(p,8,c) + 1/c^8 *chi.int(p,10,c)
    E<- chi.int(p,4,c)/4 - 1/(2*c^2)*chi.int(p,6,c) + 5/(12*c^4)*chi.int(p,8,c) - 1/(6*c^6)*chi.int(p,10,c) +1/(36*c^8)*chi.int(p,12,c) + c^4/36* chi.int2(p,0,c)
    
    integrand<-function(x,mu,sigma,c){
      p<-length(mu)
      distx<-Mahaladist(x,mu,sigma)
      central<-t(x)-mu
      return( as.numeric( C1^2 *scaledpsibiweight(distx,c)^2 *(colSums(solve(sigma,central)*mu))^4 ))
    }	
    ech<- rmvnorm(100000,mu,sigma)
    I<-mean(integrand(ech, mu=mu,sigma=sigma,c))
    ASV<- g^2/4*(g^4*I + C2^2*E - C3^2*B - C4^2 +4/p*g^2*C5^2*D )
  }else stop("This is not  valid estim value. The possible values are: 'class', 'MCD' or 'S'")
  return(ASV)
}



#----------------- Auxiliary function 

Mahaladist<-function(x,mu,Sigma){
  central<-t(x)-mu
  sqdis<-colSums(solve(Sigma,central)*central)
  dis<-sqdis^(0.5)
  return(dis)
}


